#include <ESP8266WiFi.h>
#include <Wire.h>
#include <RTClib.h>
#include <ESP8266WebServer.h>
#include <EEPROM.h>
#include <WiFiUdp.h>
#include <ESP8266HTTPClient.h>
#include <WiFiClient.h>


// ---------------- AP CONFIG ----------------
#define AP_SSID "SmartWaterLevelController"   // Username
#define AP_PASS "12345678"                    // Password


// ---------------- PIN DEFINITIONS ----------------
#define TRIG_PIN       D6
#define ECHO_PIN       D7
#define WIFI_LED_PIN   D0
#define AUTO_LED_PIN   D3
#define MANUAL_LED_PIN D4
#define RELAY_PIN      D5

// ---------------- GLOBAL OBJECTS ----------------
ESP8266WebServer server(80);
WiFiUDP udp;
unsigned int port = 4210;

#define EEPROM_SIZE 1024

#define ADDR_TANK_HEIGHT   0
#define ADDR_HIGH_LEVEL    10
#define ADDR_LOW_LEVEL     20
#define ADDR_AUTOMODE      30
#define ADDR_MOTOR_STATE   40
#define ADDR_SSID          50    // 32 bytes → till 81
#define ADDR_PASS          90    // 64 bytes → till 153
#define ADDR_TIME_FORMAT   160
#define RTC_VALID_ADDR     170
#define ADDR_DRYRUN_TIME   180   // free space, seconds
#define ADDR_SCHEDULE      300


IPAddress lastSTAIP(0, 0, 0, 0);
bool ipChangedBlinkDone = false;

bool is24HourFormat = true;   // default
bool rtcSynced = false;

bool scheduleCancelledToday = false;

bool notifyAutoOn = false;
bool notifyAutoOff = false;
bool notifyScheduleOn = false;
bool notifyScheduleOff = false;
bool notifyDryRun = false;


unsigned long notifyAutoOnTime = 0;
unsigned long notifyAutoOffTime = 0;
unsigned long notifyScheduleOnTime = 0;
unsigned long notifyScheduleOffTime = 0;
unsigned long notifyDryRunTime = 0;


int activeScheduleSlot = -1;

unsigned long lastScheduleEndMillis = 0;




RTC_DS3231 rtc;
bool rtcPrinted = false;
long timezoneOffset = 0;   // seconds

int activeScheduleDay = -1;


bool scheduleRunning = false;

// 🔔 Pump start/stop source tracking
enum PumpSource {
  SRC_NONE,
  SRC_SCHEDULE,
  SRC_MANUAL,
  SRC_AUTO,
  SRC_DRYRUN
};

PumpSource lastPumpSource = SRC_NONE;

int lastPrintedMinute = -1;
int lastCheckedMinute = -1;
unsigned long lastTimePrintMillis = 0;




// ---------- SCHEDULE MODEL ----------
#define MAX_SCHEDULES_PER_DAY 4   // ⭐ 4 slots per day

struct ScheduleSlot {
  bool enable = false;
  uint8_t onH = 0;
  uint8_t onM = 0;
  uint8_t offH = 0;
  uint8_t offM = 0;
};

struct DaySchedule {
  ScheduleSlot slot[MAX_SCHEDULES_PER_DAY];
};

DaySchedule week[7];

// -------- DAY NAMES (0 = Sunday) --------
const char* DAY_NAMES[] = {
  "Sunday",
  "Monday",
  "Tuesday",
  "Wednesday",
  "Thursday",
  "Friday",
  "Saturday"
};

String format12Hour(int h, int m) {
  String ampm = (h >= 12) ? "PM" : "AM";
  int h12 = h % 12;
  if (h12 == 0) h12 = 12;

  char buf[12];
  sprintf(buf, "%02d:%02d %s", h12, m, ampm.c_str());
  return String(buf);
}



// ---------------- TANK VARIABLES ----------------
float tankHeight = 40;
float highLevelPercent = 90;
float lowLevelPercent  = 30;
bool motorState = false;
bool autoMode = true;

float filteredLevel = 0;
int lastLevel = -1;
unsigned long lastLevelChange = 0;
unsigned long lastCheck = 0;

unsigned long dryRunTimeoutSec = 60;   // default 60 sec
bool dryRunTriggered = false;

// ---------------- ROUTER CREDENTIALS ----------------
char routerSSID[32] = "";
char routerPASS[64] = "";
bool staConfigured = false;

// ---------------- RELAY NODE CONFIG ----------------
const char* RELAY_NODE_IP = "192.168.4.2";

void sendRelayCommand(bool state) {

  
  if (WiFi.softAPgetStationNum() == 0) {
    Serial.println("❌ Wireless Relay NOT connected");
    return;
  }

  WiFiClient client;
  HTTPClient http;

  String url = String("http://") + RELAY_NODE_IP +
               "/relay?state=" + (state ? "1" : "0");

  // ✅ Simple readable status
  if (state)
    Serial.println("📶🔌 Wireless Relay ON"); 
  else
    Serial.println("📶🔌 Wireless Relay OFF");

  if (http.begin(client, url)) {
    int code = http.GET();

    if (code == 200) {
      Serial.println("✅ OK");
    } else {
      Serial.print("Relay Error: ");
      Serial.println(code);
    }

    http.end();
  } else {
    Serial.println("Relay HTTP FAILED");
  }
}


// ------------------------------------------------------
//                       ULTRASONIC
// ------------------------------------------------------
float getDistance() {
  float sum = 0;
  for (int i = 0; i < 5; i++) {
    digitalWrite(TRIG_PIN, LOW);
    delayMicroseconds(2);
    digitalWrite(TRIG_PIN, HIGH);
    delayMicroseconds(10);
    digitalWrite(TRIG_PIN, LOW);

    float duration = pulseIn(ECHO_PIN, HIGH, 25000);
    float d = duration * 0.0343 / 2.0;
    if (d < 2 || d > 500) d = tankHeight;

    sum += d;
    delay(10);
    yield();
  }
  return sum / 5;
}

float getLevelPercent() {
  float distance = getDistance();
  float level = 100 - ((distance / tankHeight) * 100);
  level = constrain(level, 0, 100);

  if (level > 95) {
    filteredLevel = 100;
    return 100;
  }

  filteredLevel = (0.8 * filteredLevel) + (0.2 * level);
  return filteredLevel;
}

// ------------------------------------------------------
//                       EEPROM
// ------------------------------------------------------
void saveSettings() {
  EEPROM.put(ADDR_TANK_HEIGHT, tankHeight);
  EEPROM.put(ADDR_HIGH_LEVEL, highLevelPercent);
  EEPROM.put(ADDR_LOW_LEVEL, lowLevelPercent);
  EEPROM.put(ADDR_AUTOMODE, autoMode);
  EEPROM.put(ADDR_MOTOR_STATE, motorState);
  EEPROM.put(ADDR_SSID, routerSSID);
  EEPROM.put(ADDR_PASS, routerPASS);
  EEPROM.put(ADDR_TIME_FORMAT, is24HourFormat);
  EEPROM.put(ADDR_DRYRUN_TIME, dryRunTimeoutSec);
  EEPROM.commit();
 
}

void loadSettings() {
  EEPROM.get(ADDR_TANK_HEIGHT, tankHeight);
  EEPROM.get(ADDR_HIGH_LEVEL, highLevelPercent);
  EEPROM.get(ADDR_LOW_LEVEL, lowLevelPercent);
  EEPROM.get(ADDR_AUTOMODE, autoMode);
  EEPROM.get(ADDR_MOTOR_STATE, motorState);
  EEPROM.get(ADDR_SSID, routerSSID);
  EEPROM.get(ADDR_PASS, routerPASS);
  EEPROM.get(ADDR_TIME_FORMAT, is24HourFormat); // ✅ ADD THIS
  EEPROM.get(ADDR_DRYRUN_TIME, dryRunTimeoutSec);

// safety
  if (dryRunTimeoutSec < 10 || dryRunTimeoutSec > 600)
    dryRunTimeoutSec = 60;


  if (tankHeight < 5 || tankHeight > 500) tankHeight = 40;
  if (highLevelPercent > 100) highLevelPercent = 90;
  if (lowLevelPercent < 0) lowLevelPercent = 30;

  staConfigured = routerSSID[0] != '\0';
}

// ================= SCHEDULE EEPROM =================
void saveSchedule(int day, int slot) {
  int addr = ADDR_SCHEDULE +
             (day * MAX_SCHEDULES_PER_DAY + slot)
             * sizeof(ScheduleSlot);

  EEPROM.put(addr, week[day].slot[slot]);
  EEPROM.commit();
}



void loadSchedules() {

  for (int d = 0; d < 7; d++) {
    for (int s = 0; s < MAX_SCHEDULES_PER_DAY; s++) {

      int addr = ADDR_SCHEDULE +
                 (d * MAX_SCHEDULES_PER_DAY + s)
                 * sizeof(ScheduleSlot);

      EEPROM.get(addr, week[d].slot[s]);

      // 🛡️ EEPROM garbage protection
      if (week[d].slot[s].onH > 23 ||
          week[d].slot[s].onM > 59 ||
          week[d].slot[s].offH > 23 ||
          week[d].slot[s].offM > 59) {

        week[d].slot[s].enable = false;
      }
    }
  }
}





void handleSetTimeFormat() {

  if (!server.hasArg("format")) {
    server.send(400, "text/plain", "NO_FORMAT");
    return;
  }

  is24HourFormat = server.arg("format") == "24";
  EEPROM.put(100, is24HourFormat);
  EEPROM.commit();

  Serial.print("⏱ Time format set: ");
  Serial.println(is24HourFormat ? "24-HOUR" : "AM/PM");

  server.send(200, "text/plain", "OK");
}


void handleGetTimeFormat() {
  server.send(200, "text/plain",
              is24HourFormat ? "24" : "12");
}

// ------------------------------------------------------
//                     MOTOR CONTROL
// ------------------------------------------------------
void motorOn() {
if (motorState) return;   // ⭐ prevent duplicate OFF

  digitalWrite(RELAY_PIN, LOW);
  motorState = true;
  dryRunTriggered = false;

  
  if (scheduleRunning)
    lastPumpSource = SRC_SCHEDULE;
  else
    lastPumpSource = SRC_MANUAL;

  sendRelayCommand(true);   // 🔥 RELAY NODE ON

  lastLevel = getLevelPercent();
  lastLevelChange = millis();

  saveSettings();
}

void motorOff() {
if (!motorState) return;   // ⭐ prevent duplicate OFF

  digitalWrite(RELAY_PIN, HIGH);
  motorState = false;

  sendRelayCommand(false);   // 🔥 RELAY NODE OFF
  saveSettings();
}

void updateMotorState(float level) {

  if (!autoMode) return;
  if (scheduleRunning) return;

  if (dryRunTriggered) return;   // Dry run active → pump block

  // ⭐ Block auto for 60 sec after schedule
  if (millis() - lastScheduleEndMillis < 60000) return;

  if (level <= lowLevelPercent && !motorState) {
    lastPumpSource = SRC_AUTO;
    motorOn();
    notifyAutoOn = true;   // ADD
    notifyAutoOnTime = millis();
  }

  if (level >= highLevelPercent && motorState) {
    lastPumpSource = SRC_AUTO;
    motorOff();
    notifyAutoOff = true;  // ADD
    notifyAutoOffTime = millis();
  }
}



void updateModeLEDs() {
  digitalWrite(AUTO_LED_PIN, autoMode ? LOW : HIGH);
  digitalWrite(MANUAL_LED_PIN, autoMode ? HIGH : LOW);
}

//-----------------------------------------------------------
//           Blink Blue LED for Change STA IP
//----------------------------------------------------------

void blinkWiFiLED4Times() {
  for (int i = 0; i < 5; i++) {
    digitalWrite(WIFI_LED_PIN, LOW);
    delay(400);
    digitalWrite(WIFI_LED_PIN, HIGH);
    delay(400);
  }
}


// ------------------------------------------------------
//                SUPER STABLE WiFi SYSTEM
// ------------------------------------------------------
extern "C" {
  #include "user_interface.h"
}

void setupWiFiStable()
{
  Serial.println("\n🚀 Setting up SUPER-STABLE WiFi...");

  WiFi.persistent(false);
  WiFi.setAutoReconnect(true);

  wifi_station_set_auto_connect(true);
  wifi_station_set_reconnect_policy(true);

  // ⭐ Modern sleep disable function
  WiFi.setSleepMode(WIFI_NONE_SLEEP);

  WiFi.mode(WIFI_AP_STA);

  // 1) Start AP on fixed channel 6
  WiFi.softAP(AP_SSID, AP_PASS, 6, false);


  Serial.print("AP Started → IP: ");
  Serial.println(WiFi.softAPIP());

  // 2) Now connect STA
  WiFi.begin(routerSSID, routerPASS);

  unsigned long start = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - start < 20000) {
    Serial.print(".");
    delay(300);
  }
  Serial.println("");

  if (WiFi.status() == WL_CONNECTED) {

    // 3) Match AP channel with router
    int ch = WiFi.channel();
    wifi_set_channel(ch);

    WiFi.softAP(AP_SSID, AP_PASS, ch, false);


    Serial.println("🌐 STA Connected!");
    Serial.print("STA IP: ");
    Serial.println(WiFi.localIP());

    Serial.print("Router Channel Detected = ");
    Serial.println(ch);

  } else {
    Serial.println("❌ STA Failed → Staying AP only.");
  }
}


void handlePing() {
  server.send(200, "text/plain", "OK");
}



// ------------------------------------------------------
//                 /setWiFi (Update Router)
// ------------------------------------------------------
void handleSetWiFi() {
  if (!server.hasArg("ssid") || !server.hasArg("pass")) {
    server.send(400, "text/plain", "Missing SSID or PASSWORD");
    return;
  }

  String ssid = server.arg("ssid");
  String pass = server.arg("pass");

  memset(routerSSID, 0, sizeof(routerSSID));
  memset(routerPASS, 0, sizeof(routerPASS));

  ssid.toCharArray(routerSSID, sizeof(routerSSID));
  pass.toCharArray(routerPASS, sizeof(routerPASS));

  staConfigured = true;
  saveSettings();

  server.send(200, "text/plain", "OK");

  ESP.restart();
}

// ------------------------------------------------------
//                     HTTP ROUTES
// ------------------------------------------------------
bool isAPModeOnly() {
  IPAddress clientIP = server.client().remoteIP();

  // If app is connected to NodeMCU hotspot (192.168.4.x)
  if (clientIP[0] == 192 && clientIP[1] == 168 && clientIP[2] == 4) {
    return true;   // AP mode
  }

  return false;    // Router mode
}


void handleGetSchedule() {

  String data = "";

  for (int d = 0; d < 7; d++) {
    for (int s = 0; s < MAX_SCHEDULES_PER_DAY; s++) {

      ScheduleSlot &sl = week[d].slot[s];
      if (!sl.enable) continue;

      data += String(d) + "," +
              String(s) + ",1," +
              sl.onH + "," + sl.onM + "," +
              sl.offH + "," + sl.offM + "|";
    }
  }

  if (data.endsWith("|"))
    data.remove(data.length() - 1);

  server.send(200, "text/plain", data);
}




void handleMotorState() {
  if (dryRunTriggered)
    server.send(200, "text/plain", "DryRun");
  else
    server.send(200, "text/plain", motorState ? "1" : "0");
}

void handleLevel() {
  
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  float level = getLevelPercent();

  
  updateMotorState(level);

  server.send(200, "text/plain", String(level, 1));
}


void handleMotorOn() {
  
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  
  if (autoMode == true) {
    server.send(403, "text/plain", "MANUAL_REQUIRED");
    return;
  }

  motorOn();
  server.send(200, "text/plain", "1");
}


void handleMotorOff() {
  
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  
  if (autoMode == true) {
    server.send(403, "text/plain", "MANUAL_REQUIRED");
    return;
  }


  if (scheduleRunning) {
    Serial.println("⛔ Manual Pump OFF → Schedule FORCE END");

    scheduleRunning   = false;
    activeScheduleDay = -1;
    lastCheckedMinute = -1;   // ⭐ VERY IMPORTANT
    scheduleCancelledToday = true;   // ⭐⭐ KEY LINE ⭐⭐
  }

  lastPumpSource = SRC_MANUAL;
  motorOff();
  server.send(200, "text/plain", "0");
}

void handleSetSchedule() {

  // ⭐ User manually updated schedule → allow today again
  scheduleCancelledToday = false;
  
int d    = server.arg("day").toInt();
int slot = server.arg("slot").toInt();   // ⭐ MUST

if (slot < 0 || slot >= MAX_SCHEDULES_PER_DAY) {
  server.send(400, "text/plain", "INVALID_SLOT");
  return;
}

ScheduleSlot &sl = week[d].slot[slot];

sl.enable = server.arg("enable") == "1";
sl.onH    = server.arg("onH").toInt();
sl.onM    = server.arg("onM").toInt();
sl.offH   = server.arg("offH").toInt();
sl.offM   = server.arg("offM").toInt();

saveSchedule(d, slot);




  // 🔁 FORCE END OLD SCHEDULE IF RUNNING
  if (scheduleRunning) {
    motorOff();
    scheduleRunning   = false;
    lastCheckedMinute = -1;

  Serial.println("⛔ Old schedule force-END (reschedule)");
}


// ✅ FINAL MULTI-LINE CONFIRMATION PRINT (APP → CONTROLLER)
if (sl.enable) {

  Serial.print("📅 ");
  Serial.print(DAY_NAMES[d]);
  Serial.print(" Schedule Set (Slot ");
  Serial.print(slot);
  Serial.println(")");

  Serial.print("   🟢 Pump ON : ");
  Serial.println(format12Hour(sl.onH, sl.onM));

  Serial.print("   🔴 Pump OFF: ");
  Serial.println(format12Hour(sl.offH, sl.offM));
}

else {

  Serial.print("❌ ");
  Serial.print(DAY_NAMES[d]);
  Serial.print(" PUMP SCHEDULE DISABLED (Slot ");
  Serial.print(slot);
  Serial.println(")");
}

  
  lastCheckedMinute = -1;
  lastPrintedMinute = -1;
  scheduleCancelledToday = false;

  server.send(200, "text/plain", "OK");


}



void handleUpdate() {
  
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  if (server.hasArg("height")) {

    tankHeight = server.arg("height").toFloat();

    // ✅ confirmation print ONLY
    Serial.print("✅ Tank Height Saved in EEPROM: ");
    Serial.print(tankHeight);
    Serial.println(" cm");
}

  if (server.hasArg("low"))    lowLevelPercent = server.arg("low").toFloat();
  if (server.hasArg("high"))   highLevelPercent = server.arg("high").toFloat();
  if (server.hasArg("mode")) {
  bool newMode = (server.arg("mode") == "1");

  // ⭐ AUTO → MANUAL switch detect
  if (autoMode == true && newMode == false) {

    Serial.println("🔁 Switching AUTO → MANUAL");

    // Stop auto immediately
    scheduleRunning = false;
    activeScheduleDay = -1;
    lastCheckedMinute = -1;

    // Pump ON ho to OFF kar do
    if (motorState) {
      lastPumpSource = SRC_MANUAL;
      motorOff();
    }
  }

  autoMode = newMode;
}


  updateModeLEDs();
  saveSettings();

  server.send(200, "text/plain", "OK");
}



void handleGetSettings() {
  
  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  String data =
    String(tankHeight) + "," +
    String(lowLevelPercent) + "," +
    String(highLevelPercent) + "," +
    (autoMode ? "1" : "0") + "," +
    (motorState ? "1" : "0") + "," +
    String(dryRunTimeoutSec);


  server.send(200, "text/plain", data);
}


void handleClearDryRun() {
  dryRunTriggered = false;
  scheduleCancelledToday = false;
  lastLevelChange = millis();   // ⭐ ADD THIS
  motorOff();
  server.send(200, "text/plain", "OK");
}


// ================= SCHEDULER CHECK (SAFE VERSION) =================
void checkSchedule() {

  if (!rtcSynced) return;   // 🔒 RTC NOT READY → EXIT

  DateTime now = rtc.now();

  // ❌ INVALID RTC DATA PROTECTION
  if (now.hour() > 23 || now.minute() > 59) return;

  // ❌ RTC invalid / battery garbage protection
  if (now.year() < 2024) {
    return;   // RTC not yet synced
  }

  int day = now.dayOfTheWeek();
  int currentMinute = now.hour() * 60 + now.minute();

  // ⭐ RESET GUARD AT DAY CHANGE
  static int lastDay = -1;
  if (day != lastDay) {
    scheduleCancelledToday = false;
    lastDay = day;
  }

// ⏱ PRINT EVERY 10 SECONDS (SAFE)
static int lastPrintedSecond = -1;

if (now.second() % 10 == 0 && now.second() != lastPrintedSecond) {
  Serial.printf("⏱ Current Time: %s %02d:%02d:%02d\n",
                DAY_NAMES[day],
                now.hour(),
                now.minute(),
                now.second());

  lastPrintedSecond = now.second();
}


  // ---- PER MINUTE GUARD ----
  if (currentMinute == lastCheckedMinute) return;
  lastCheckedMinute = currentMinute;

  // ======================================================
  // 🔁 MULTIPLE SCHEDULES PER DAY (SLOT SYSTEM)
  // ======================================================
  for (int s = 0; s < MAX_SCHEDULES_PER_DAY; s++) {

    ScheduleSlot &sl = week[day].slot[s];
    if (!sl.enable) continue;

    int onMinute  = sl.onH  * 60 + sl.onM;
    int offMinute = sl.offH * 60 + sl.offM;

    // ▶️ SCHEDULE START
   if (currentMinute >= onMinute &&
       currentMinute < offMinute &&
       !scheduleRunning &&
       !scheduleCancelledToday) {

     lastPumpSource = SRC_SCHEDULE;
     motorOn();
     scheduleRunning = true;
     activeScheduleDay = day;
     activeScheduleSlot = s;   // ⭐ important

     notifyScheduleOn = true;   // ADD HERE
     notifyScheduleOnTime = millis();


     Serial.printf("▶️ SCHEDULE START → %s (SLOT %d) %02d:%02d\n",
                DAY_NAMES[day], s, sl.onH, sl.onM);
  }


    // ⏹️ SCHEDULE END
    if (scheduleRunning &&
    activeScheduleSlot == s &&
    currentMinute >= offMinute) {

  lastPumpSource = SRC_SCHEDULE;
  motorOff();

  notifyScheduleOff = true;   // ADD HERE
  notifyScheduleOffTime = millis();

  scheduleRunning = false;
  activeScheduleDay = -1;
  activeScheduleSlot = -1;   // ⭐ important

  lastScheduleEndMillis = millis();   // ⭐ ADD THIS LINE


  scheduleCancelledToday = false;
  lastCheckedMinute = -1;

  Serial.printf("⏹️ SCHEDULE END → %s (SLOT %d)\n",
                DAY_NAMES[day], s);
}

  }
}



 void safeRTCInit() {

  Wire.beginTransmission(0x68);
  Wire.write(0x0F);   // status register
  if (Wire.endTransmission() != 0) {
    Serial.println("❌ RTC not responding");
    return;
  }

  Wire.requestFrom(0x68, 1);
  if (!Wire.available()) {
    Serial.println("❌ RTC status read fail");
    return;
  }

  byte status = Wire.read();

  if (status & 0x80) {
    Serial.println("⚠️ RTC OSF set (time may be invalid)");
  } else {
    Serial.println("✅ RTC oscillator running");
  }
}



void handleScheduleStatus() {

  String src = "UNKNOWN";

  if (lastPumpSource == SRC_SCHEDULE) src = "SCHEDULE";
  else if (lastPumpSource == SRC_AUTO) src = "AUTO";
  else if (lastPumpSource == SRC_MANUAL) src = "MANUAL";
  else if (lastPumpSource == SRC_DRYRUN) src = "DRYRUN";

  // ---------- MOTOR OFF ----------
  if (!motorState) {
    server.send(200, "text/plain",
                "STOPPED,SOURCE=" + src);
    return;
  }

  // ---------- MOTOR RUNNING ----------
  int day = rtc.now().dayOfTheWeek();

  server.send(200, "text/plain",
              "RUNNING,DAY=" + String(day) + ",SOURCE=" + src);
}


void handleClearSchedule() {

  Serial.println("🧹 CLEAR ALL SCHEDULES REQUEST RECEIVED");

  for (int i = 0; i < 7; i++) {
  for (int s = 0; s < MAX_SCHEDULES_PER_DAY; s++) {

    week[i].slot[s].enable = false;
    week[i].slot[s].onH = 0;
    week[i].slot[s].onM = 0;
    week[i].slot[s].offH = 0;
    week[i].slot[s].offM = 0;

    saveSchedule(i, s);
  }
}


  // 🔒 Runtime safety
  scheduleRunning = false;
  activeScheduleDay = -1;
  lastCheckedMinute = -1;
  scheduleCancelledToday = false;

  Serial.println("✅ ALL SCHEDULES CLEARED");

  server.send(200, "text/plain", "ALL_CLEARED");
}



void handleSetTime() {

  if (isAPModeOnly()) {
    server.send(403, "text/plain", "AP_MODE_LIMITED");
    return;
  }

  if (!server.hasArg("plain")) {
    server.send(400, "text/plain", "NO_JSON");
    return;
  }

  String body = server.arg("plain");

  int eIdx = body.indexOf("\"epoch\"");
  int tIdx = body.indexOf("\"tz\"");

  if (eIdx < 0 || tIdx < 0) {
    server.send(400, "text/plain", "INVALID_JSON");
    return;
  }

  unsigned long epoch =
      body.substring(body.indexOf(":", eIdx) + 1).toInt();

  long tz =
      body.substring(body.indexOf(":", tIdx) + 1).toInt();

      Serial.println("📲 TIME RECEIVED FROM APP");
      Serial.print("   Epoch: ");
      Serial.println(epoch);
      Serial.print("   TZ Offset: ");
      Serial.println(tz);


  if (epoch < 1000000000) {
    server.send(400, "text/plain", "INVALID_EPOCH");
    return;
  }

  // ===== FINAL & RELIABLE RTC WRITE =====
  DateTime localTime(epoch + tz);

  // 🔧 HARD RESET I2C BUS (MOST IMPORTANT LINE)
  Wire.begin(D2, D1);
  delay(50);

  rtc.adjust(localTime);
  delay(100);

  DateTime verify = rtc.now();

  if (verify.year() >= 2024 &&
      verify.year() <= 2100 &&
      verify.hour() <= 23 &&
      verify.minute() <= 59) {

    rtcSynced = true;

    byte flag = 0xA5;
    EEPROM.put(RTC_VALID_ADDR, flag);
    EEPROM.commit();

    lastPrintedMinute = -1;
    lastCheckedMinute = -1;
    scheduleRunning = false;

    Serial.println("✅ RTC TIME SAVED SUCCESSFULLY");
    Serial.printf("   RTC Now: %04d-%02d-%02d %02d:%02d:%02d\n",
              verify.year(), verify.month(), verify.day(),
              verify.hour(), verify.minute(), verify.second());


    server.send(200, "text/plain", "TIME_SET_OK");
    return;
  }

  Serial.println("❌ RTC TIME SAVE FAILED");

  rtcSynced = false;
  server.send(500, "text/plain", "RTC_FAIL");
}




void handleNotifyAll() {

  String json = "{";

  json += "\"autoOn\":" + String(notifyAutoOn ? 1 : 0) + ",";
  json += "\"autoOff\":" + String(notifyAutoOff ? 1 : 0) + ",";
  json += "\"scheduleOn\":" + String(notifyScheduleOn ? 1 : 0) + ",";
  json += "\"scheduleOff\":" + String(notifyScheduleOff ? 1 : 0) + ",";
  json += "\"dryRun\":" + String(notifyDryRun ? 1 : 0);

  json += "}";

  server.send(200, "application/json", json);

}


void handleSetDryRunSensitivity() {

  if (!server.hasArg("sec")) {
    server.send(400, "text/plain", "NO_VALUE");
    return;
  }

  dryRunTimeoutSec = server.arg("sec").toInt();

  if (dryRunTimeoutSec < 10) dryRunTimeoutSec = 10;
  if (dryRunTimeoutSec > 600) dryRunTimeoutSec = 600;

  EEPROM.put(ADDR_DRYRUN_TIME, dryRunTimeoutSec);
  EEPROM.commit();

  Serial.print("🔥 Dry Run Sensitivity set to ");
  Serial.print(dryRunTimeoutSec);
  Serial.println(" seconds");

  server.send(200, "text/plain", "OK");
}





// ------------------------------------------------------
//                        SETUP
// ------------------------------------------------------
void setup() {
  Serial.begin(115200);
  EEPROM.begin(EEPROM_SIZE);

  // -------- I2C INIT --------
  Wire.begin(D2, D1);
  Wire.setClock(100000);   // ⭐ VERY IMPORTANT
  delay(2000);

  // -------- RTC ADDRESS CHECK --------
  Wire.beginTransmission(0x68);
  byte err = Wire.endTransmission();
  Serial.print("RTC Addr 0x68 = ");
  Serial.println(err == 0 ? "FOUND" : "NOT FOUND");

  safeRTCInit();   // ✅ ADD THIS

 

  // -------- RTC LIB INIT --------
  if (!rtc.begin()) {
    Serial.println("❌ RTC not found (RTClib)");
  } else {
    Serial.println("✅ RTC begin OK");
  }

  // -------- PRINT RAW RTC --------
  delay(200);                 // ⭐ let I2C settle
  DateTime now = rtc.now();   // ⭐ fresh stable read

  Serial.printf("RTC RAW: %02d:%02d:%02d %02d-%02d-%04d\n",
              now.hour(), now.minute(), now.second(),
              now.day(), now.month(), now.year());


              

  // -------- LOAD STORED DATA --------
  loadSettings();
  loadSchedules();

  now = rtc.now();   // ⭐ re-read RTC just before validation

  byte rtcFlag = 0;
  EEPROM.get(RTC_VALID_ADDR, rtcFlag);


if (rtcFlag == 0xA5 &&
    now.year() >= 2024 &&
    now.year() <= 2100 &&
    now.hour() <= 23 &&
    now.minute() <= 59) {

  rtcSynced = true;
  Serial.println("🔓 RTC restored from battery");

} else {
  rtcSynced = false;
  Serial.println("⛔ RTC invalid – waiting for app time");
}


  Serial.print("⏱ Stored Time Format: ");
  Serial.println(is24HourFormat ? "24-HOUR" : "AM/PM");

  // -------- PIN MODES --------
  pinMode(TRIG_PIN, OUTPUT);
  pinMode(ECHO_PIN, INPUT);
  pinMode(RELAY_PIN, OUTPUT);
  pinMode(WIFI_LED_PIN, OUTPUT);
  pinMode(AUTO_LED_PIN, OUTPUT);
  pinMode(MANUAL_LED_PIN, OUTPUT);

  // 🔒 Default relay OFF (active LOW)
  digitalWrite(RELAY_PIN, HIGH);
  if (motorState) {
    digitalWrite(RELAY_PIN, LOW);
  }

  updateModeLEDs();

  // -------- WIFI --------
  setupWiFiStable();
  udp.begin(port);

  // -------- ROUTES --------
  server.on("/level", handleLevel);
  server.on("/motor/state", handleMotorState);
  server.on("/motor/on", handleMotorOn);
  server.on("/motor/off", handleMotorOff);
  server.on("/update", handleUpdate);
  server.on("/getSettings", handleGetSettings);
  server.on("/getSchedule", handleGetSchedule);
  server.on("/setWiFi", handleSetWiFi);
  server.on("/setSchedule", handleSetSchedule);
  server.on("/clearDryRun", handleClearDryRun);
  server.on("/setTime", HTTP_POST, handleSetTime);
  server.on("/setTimeFormat", handleSetTimeFormat);
  server.on("/getTimeFormat", handleGetTimeFormat);
  server.on("/scheduleStatus", handleScheduleStatus);
  server.on("/clearSchedule", handleClearSchedule);
  server.on("/notify/all", handleNotifyAll);
  server.on("/setDryRun", handleSetDryRunSensitivity);
  server.on("/ping", handlePing); 
  server.on("/getRouterSSID", []() {
  server.send(200, "text/plain", WiFi.SSID());
  });

  server.begin();
  Serial.println("HTTP Server started");
}


// ------------------------------------------------------
//                        LOOP
// ------------------------------------------------------
void loop() {
  server.handleClient();

  // -------- UDP Discovery --------
  int size = udp.parsePacket();
  if (size) {
    char msg[50];
    int len = udp.read(msg, 49);
    msg[len] = 0;

    if (String(msg).startsWith("DISCOVER_NODEMCU")) {
      IPAddress ip = (WiFi.status() == WL_CONNECTED) ? WiFi.localIP() : WiFi.softAPIP();
      String r = "NODEMCU_OK,IP=" + ip.toString() + ",ID=Tank1";

      udp.beginPacket(udp.remoteIP(), udp.remotePort());
      udp.print(r);
      udp.endPacket();
    }
  }
  
// -------- STA IP CHANGE DETECTION --------
if (WiFi.status() == WL_CONNECTED) {

  IPAddress currentIP = WiFi.localIP();

  if (currentIP != lastSTAIP && !ipChangedBlinkDone) {
    Serial.print("🔁 STA IP Changed: ");
    Serial.println(currentIP);

    blinkWiFiLED4Times();          // 🔵 4 blinks once
    lastSTAIP = currentIP;
    ipChangedBlinkDone = true;     // 🔒 lock blinking
  }

} else {
  // STA disconnected → allow blink next time
  ipChangedBlinkDone = false;
}


  //-----------------------------------------------------------
  // -------- DRY RUN DETECTION --------
  //-------------------------------------------------------------
  unsigned long now = millis();

  if (motorState && !dryRunTriggered) {
    float level = getLevelPercent();

    if (abs(level - lastLevel) > 1) {
      lastLevel = level;
      lastLevelChange = now;
    } 
  else if (now - lastLevelChange > (dryRunTimeoutSec * 1000UL)) {

    Serial.println("🚨 DRY RUN DETECTED → PUMP FORCE STOP");
    
    lastPumpSource = SRC_DRYRUN;
    dryRunTriggered = true;

    notifyDryRun = true;   // ⭐ ADD THIS LINE
    notifyDryRunTime = millis();

    
    motorOff();   // relay off

    // 🔴 SAME AS MANUAL PUMP OFF
    scheduleRunning = false;
    activeScheduleDay = -1;
    lastCheckedMinute = -1;
    scheduleCancelledToday = true;   // ⭐⭐ MOST IMPORTANT LINE ⭐⭐
  }

  }

  // -------- WiFi LED --------
  if (WiFi.status() == WL_CONNECTED || WiFi.softAPgetStationNum() > 0)
    digitalWrite(WIFI_LED_PIN, LOW);
  else
    digitalWrite(WIFI_LED_PIN, (now / 500) % 2 ? HIGH : LOW);

  checkSchedule();

  unsigned long nowMillis = millis();

if (notifyAutoOn && nowMillis - notifyAutoOnTime > 10000)
    notifyAutoOn = false;

if (notifyAutoOff && nowMillis - notifyAutoOffTime > 10000)
    notifyAutoOff = false;

if (notifyScheduleOn && nowMillis - notifyScheduleOnTime > 10000)
    notifyScheduleOn = false;

if (notifyScheduleOff && nowMillis - notifyScheduleOffTime > 10000)
    notifyScheduleOff = false;

if (notifyDryRun && nowMillis - notifyDryRunTime > 10000)
    notifyDryRun = false;


}

